/**
 *  Copyright © 2022-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.telko;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyVetoException;
import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JInternalFrame;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.nexuswob.gui.InputDocument;
import net.javacomm.client.gui.MultilingualServiceButton;
import net.javacomm.client.resource.Resource;
import net.javacomm.multilingual.MultilingualLabel;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.share.Constants;
import net.javacomm.window.manager.Control;
import net.javacomm.window.manager.WindowManagerInternalFrame;



@SuppressWarnings("serial")
public class JTelefonkonferenzErstellenFrame extends WindowManagerInternalFrame {

  private final static Logger log = LogManager.getLogger(JTelefonkonferenzErstellenFrame.class);

  public final static int SCREEN_WIDTH = 600;
  public final static int SCREEN_HEIGHT = 390;
  public final static int BREITE = Constants.LEN_KONFERENZ_BESCHREIBUNG;
  private GridBagLayout gridbag = new GridBagLayout();
  private GridBagConstraints con = new GridBagConstraints();
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private MultilingualLabel multilingualKonferenzname = new MultilingualLabel(KEY.LABEL_KONFERENZNAME);
  private JTextField textfieldKonferenzname = new JTextField();
  private JSeparator separator = new JSeparator();
  private MultilingualServiceButton servicebutton = new MultilingualServiceButton(
      KEY.STRING_CREATE, KEY.BUTTON_SCHLIESSEN
  );
  private JTextArea textareaBeschreibung = new JTextArea();
  private JScrollPane scrollpane = new JScrollPane(textareaBeschreibung);

  private MultilingualLabel labelBeschreibung = new MultilingualLabel(KEY.STRING_BESCHREIBUNG);
  private MultilingualLabel labeltext = new MultilingualLabel(KEY.LABEL_DIE_BESCHREIBUNG_WIRD);
  private InputDocument inputdocument = new InputDocument(Constants.LEN_KONFERENZNAME);
  private DocumentListener documentAdapter = new DocumentListener() {

    private String text;

    private void check(DocumentEvent event) {
      try {
        Document doc = event.getDocument();
        int last = doc.getEndPosition().getOffset();
        text = doc.getText(0, last);
        if (text.length() >= 2) {
          servicebutton.setEnabledButton1();
        }
        else {
          servicebutton.setDisabledButton1();
        }
      }
      catch (BadLocationException e) {
        log.error(e.getMessage(), e);
      }
    }



    @Override
    public void insertUpdate(DocumentEvent event) {
      check(event);
    }



    @Override
    public void removeUpdate(DocumentEvent event) {
      check(event);
    }



    @Override
    public void changedUpdate(DocumentEvent event) {
      check(event);
    }

  };

  public JTelefonkonferenzErstellenFrame() {
    setClosable(true);
    setResizable(true);
    setDefaultCloseOperation(JInternalFrame.DISPOSE_ON_CLOSE);
    setSize(
        new Dimension(
            JTelefonkonferenzErstellenFrame.SCREEN_WIDTH, JTelefonkonferenzErstellenFrame.SCREEN_HEIGHT
        )
    );
    addInternalFrameListener(new InternalFrameAdapter() {

      @Override
      public void internalFrameClosing(InternalFrameEvent event) {
        changes
            .firePropertyChange(JTelefonkonferenzErstellenFrame.class.getName(), Control.NULL, Control.CLOSE);
      }
    });

    this.setMinimumSize(new Dimension(SCREEN_WIDTH, SCREEN_HEIGHT));
    this.setMaximumSize(new Dimension(SCREEN_WIDTH + 160, SCREEN_HEIGHT + 90));

    init();

  }



  private void init() {
    setLayout(gridbag);

    con.gridx = 0;
    con.gridy = 0;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.insets = new Insets(12, 0, 0, 0);
    con.fill = GridBagConstraints.NONE;
    con.anchor = GridBagConstraints.CENTER;
    con.weightx = 0.0;
    con.weighty = 0.0;
    gridbag.setConstraints(multilingualKonferenzname, con);
    add(multilingualKonferenzname);

    con.gridx = 0;
    con.gridy = 1;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.insets = new Insets(0, 12, 0, 0);
    con.fill = GridBagConstraints.HORIZONTAL;
    con.anchor = GridBagConstraints.CENTER;
    con.weightx = 0.0;
    con.weighty = 0.0;
    gridbag.setConstraints(textfieldKonferenzname, con);
    add(textfieldKonferenzname);
    textfieldKonferenzname.setDocument(inputdocument);
    textfieldKonferenzname.setMaximumSize(new Dimension(BREITE, 32));
    inputdocument.addDocumentListener(documentAdapter);
    textfieldKonferenzname.getInputMap(JComponent.WHEN_FOCUSED)
        .put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0), "ENTER");
    textfieldKonferenzname.getActionMap().put("ENTER", new AbstractAction() {

      @Override
      public void actionPerformed(ActionEvent event) {
        String text = textfieldKonferenzname.getText();
        if (text.length() == 0) return;
        servicebutton.pressButton_1();
      }

    });

    con.gridx = 0;
    con.gridy++;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.insets = new Insets(6, 0, 0, 0);
    con.fill = GridBagConstraints.NONE;
    con.anchor = GridBagConstraints.CENTER;
    con.weightx = 0.0;
    con.weighty = 0.0;
    gridbag.setConstraints(labelBeschreibung, con);
    add(labelBeschreibung);

    con.gridx = 0;
    con.gridy++;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.insets = new Insets(0, 12, 0, 0);
    con.fill = GridBagConstraints.NONE;
    con.anchor = GridBagConstraints.NORTH;
    con.weightx = 0.0;
    con.weighty = 0.0;
    gridbag.setConstraints(scrollpane, con);
    add(scrollpane);
    scrollpane.setMaximumSize(new Dimension(BREITE, 180));
    scrollpane.setMinimumSize(new Dimension(BREITE, 180));
    scrollpane.setPreferredSize(new Dimension(BREITE, 180));
    textareaBeschreibung.setDocument(new InputDocument(Constants.LEN_KONFERENZ_BESCHREIBUNG));
    textareaBeschreibung.setLineWrap(true);
    textareaBeschreibung.setWrapStyleWord(true);
    textareaBeschreibung.getInputMap(JComponent.WHEN_FOCUSED)
        .put(KeyStroke.getKeyStroke(KeyEvent.VK_TAB, 0), "TAB");
    textareaBeschreibung.getActionMap().put("TAB", new AbstractAction() {

      @Override
      public void actionPerformed(ActionEvent event) {
        textareaBeschreibung.transferFocus();
      }

    });

    con.gridx++;
//  con.gridy++;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.insets = new Insets(0, 12, 12, 12);
    con.fill = GridBagConstraints.HORIZONTAL;
    con.anchor = GridBagConstraints.SOUTH;
    con.weightx = 1.0;
    con.weighty = 0.0;
    gridbag.setConstraints(labeltext, con);
    add(labeltext);
    labeltext.setIcon(new ImageIcon(getClass().getResource(Resource.LEFT_ARROW_34x26)));

    con.gridx = 0;
    con.gridy++;
    con.gridwidth = 2;
    con.insets = new Insets(24, 12, 12, 12);
    con.fill = GridBagConstraints.HORIZONTAL;
    con.anchor = GridBagConstraints.CENTER;
    con.weightx = 0.0;
    con.weighty = 0.0;
    gridbag.setConstraints(separator, con);
    add(separator);

    con.gridx = 0;
    con.gridy++;
    con.gridwidth = 2;
    con.insets = new Insets(0, 0, 12, 12);
    con.fill = GridBagConstraints.NONE;
    con.anchor = GridBagConstraints.NORTHEAST;
    con.weightx = 0.0;
    con.weighty = 1.0;
    gridbag.setConstraints(servicebutton, con);
    add(servicebutton);
    servicebutton.setDisabledButton1();
    servicebutton.addButtonListener((event) -> {

      Control value = (Control) event.getNewValue();
      switch(value) {
        case BUTTON_1:
          changes.firePropertyChange(
              JTelefonkonferenzErstellenFrame.class.getName(), Control.NULL, Control.CREATE
          );
          break;
        case BUTTON_2:
          try {
            this.setClosed(true);
          }
          catch (PropertyVetoException e) {}
          break;
        default:
          break;
      }

    });

  }



  /**
   * Über diesen Listener wird mit der Außenwelt kommuniziert.
   * 
   * @param listener
   *                 ein Listener
   */
  public synchronized void addErstellenListener(PropertyChangeListener listener) {
    changes.addPropertyChangeListener(listener);
  }



  /**
   * Der Listener wird abgestellt.
   * 
   * @param listener
   *                 ein Listener
   */
  public synchronized void removeErstellenListener(PropertyChangeListener listener) {
    changes.removePropertyChangeListener(listener);
  }



  /**
   * Alle Listener werden entfernt.
   * 
   */
  public synchronized void removeAllListener() {
    for (PropertyChangeListener pl : changes.getPropertyChangeListeners()) {
      removeErstellenListener(pl);
    }
    servicebutton.removeAllListener();
    inputdocument.removeDocumentListener(documentAdapter);
    servicebutton.removeAllListener();

  }



  /**
   * Der Konferenzname wird zurückgegeben.
   * 
   * @return der Konferenzname ist ein Text
   */
  public String getKonferenzname() {
    return textfieldKonferenzname.getText();
  }



  /**
   * Die Konferenzbeschreibung wird zurückgegeben.
   * 
   * @return ein Beschreibungstext
   */
  public String getBeschreibung() {
    return textareaBeschreibung.getText();
  }



  @Override
  public void activated() {
    try {
      setSelected(true);
    }
    catch (PropertyVetoException e) {}
  }



  @Override
  public void setFrameTitleId(String id) {
    setTitle(id);
  }



  @Override
  public void setLanguage(ISO639 code) {
    multilingualKonferenzname.setLanguage(code);
    servicebutton.setLanguage(code);
    servicebutton.setMnemonicButton2(-1);
    labeltext.setLanguage(code);
    labelBeschreibung.setLanguage(code);

    setLanguage(KEY.FRAME_TELEFONKONFERENZ_ERSTELLEN, code); // Fenstertitel

  }



  @Override
  public void setBackgroundGradient(Color top, Color bottom) {
    windowManager.setBackgroundGradient(top, bottom);
  }

}
