/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.info;

import java.awt.Color;
import java.awt.FontMetrics;
import java.awt.GradientPaint;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.net.URL;
import javax.swing.ImageIcon;
import javax.swing.JDesktopPane;
import net.javacomm.client.config.schema.Sorte;
import net.javacomm.client.environment.GUI;
import net.javacomm.client.resource.Resource;



@SuppressWarnings("serial")
public class Desktop extends JDesktopPane {

  private Image image;
  private int breite, hoehe;
  private Color color;
  private BufferedImage fullscreen;
  private boolean hidden = true;
  private Color von;
  private Color bis;
  private GradientPaint gp;
  private FontMetrics fm;

  public Desktop() {
    super();
    color = Resource.JQUERY_VANILLE;
    von = (Color) GUI.vanille.getLookAndFeel().getDefaults().get("menubar.top");
    bis = (Color) GUI.vanille.getLookAndFeel().getDefaults().get("menubar.bottom");
    hiddenProzent();
  }



  public void clear() {
    image = null;
    repaint();
  }



  @Override
  public void paintComponent(Graphics g) {
    super.paintComponent(g);
    BufferedImage bufferedImage = new BufferedImage(
        getSize().width, getSize().height, BufferedImage.TYPE_INT_ARGB_PRE
    );

    // zeichne den Progress, wenn notwendig

    Graphics2D g2 = bufferedImage.createGraphics();
    g2.setColor(color);
    g2.fillRect(0, 0, bufferedImage.getWidth(), bufferedImage.getHeight());
    if (!hidden) g2.drawImage(fullscreen, 0, 0, this);
    g2.drawImage(image, (this.getSize().width - breite) >> 1, (this.getSize().height - hoehe) >> 1, this);
    g2.dispose();
    g.drawImage(bufferedImage, 0, 0, this);

  }



  /**
   * Die Prozentangabe wird versteckt.
   * 
   * 
   */
  public void hiddenProzent() {
    hidden = true;
    repaint();
  }



  public void setPicture(URL url) {
    image = new ImageIcon(url).getImage();
    breite = image.getWidth(null);
    hoehe = image.getHeight(null);
    repaint();
  }



  public int getLineHeight() {

    fm = getFontMetrics(GUI.prozentFont16);
    return fm.getAscent() + fm.getDescent() + fm.getLeading();
  }



  /**
   * Das Farbschema wird gesetzt.
   * 
   * 
   * @param sorte
   *              eine Eissorte
   */
  public void setSorte(Sorte sorte) {
    switch(sorte) {
      case MOKKA:
        von = (Color) GUI.mokka.getLookAndFeel().getDefaults().get("menubar.top");
        bis = (Color) GUI.mokka.getLookAndFeel().getDefaults().get("menubar.bottom");
        color = Resource.JQUERY_MOKKA;
        break;
      case BLAUBEERE:
        von = (Color) GUI.blaubeere.getLookAndFeel().getDefaults().get("menubar.top");
        bis = (Color) GUI.blaubeere.getLookAndFeel().getDefaults().get("menubar.bottom");
        color = Resource.JQUERY_HELLBLAU;
        break;
      case ERDBEERE:
        von = (Color) GUI.erdbeere.getLookAndFeel().getDefaults().get("menubar.top");
        bis = (Color) GUI.erdbeere.getLookAndFeel().getDefaults().get("menubar.bottom");
        color = Resource.JQUERY_ERDBEERE;
        break;
      case JOGHURT:
        von = (Color) GUI.joghurt.getLookAndFeel().getDefaults().get("menubar.top");
        bis = (Color) GUI.joghurt.getLookAndFeel().getDefaults().get("menubar.bottom");
        color = Color.WHITE;
        break;
      case VANILLE:
        von = (Color) GUI.vanille.getLookAndFeel().getDefaults().get("menubar.top");
        bis = (Color) GUI.vanille.getLookAndFeel().getDefaults().get("menubar.bottom");
        color = Resource.JQUERY_VANILLE;
        break;
      case ZITRONE:
        von = (Color) GUI.zitrone.getLookAndFeel().getDefaults().get("menubar.top");
        bis = (Color) GUI.zitrone.getLookAndFeel().getDefaults().get("menubar.bottom");
        color = Resource.JQUERY_LEMON;
        break;
      default:
        throw new UnsupportedOperationException(
            sorte.toString() + " - diese Eissorte wird nicht unterstützt"
        );
    }
  }



  /**
   * Zeige den Fortschrittsbalken bis n-Prozent an. Der Fortschritt liegt im
   * Bereich [0..100].
   * 
   * @param prozent
   *                der aktuelle Fortschrit
   */
  public void showBar(int prozent) {
    if ((0 > prozent) || (prozent > 100))
      throw new RuntimeException(prozent + " - liegt nicht im Intervall [0..100]");

    final int einfach = this.getLineHeight();
    final int height = einfach << 1;
    hidden = false;
    int desktopWidth = getSize().width;

    int desktopHeight = getSize().height;
    int currentWidth = prozent * desktopWidth / 100;

    fullscreen = new BufferedImage(desktopWidth, desktopHeight, BufferedImage.TYPE_INT_ARGB_PRE);

    Graphics2D g2Prozentimage = fullscreen.createGraphics();
    g2Prozentimage
        .setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_GASP); // sehrgut

    gp = new GradientPaint(0, 0, Color.BLACK, currentWidth, 0, Resource.JQUERY_GREEN);
    g2Prozentimage.setPaint(gp);
    g2Prozentimage.fillRect(0, 0, currentWidth, height);

    // Prozentangabe wird blau gemalt
    g2Prozentimage.setColor(Resource.JQUERY_HELLBLAU);
    g2Prozentimage.setFont(GUI.prozentFont16);
    g2Prozentimage.drawString(String.valueOf(prozent) + "%", 40, ((height + einfach) >> 1) - fm.getDescent());

    g2Prozentimage.drawImage(fullscreen, 0, 0, this);
    g2Prozentimage.dispose();
    repaint();
  }

}
