/**
 *  Copyright © 2021-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.gui;

import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JToolTip;
import javax.swing.ToolTipManager;
import org.nexuswob.gui.ArrowTooltip;
import org.nexuswob.gui.JToolTipBorder;
import net.javacomm.client.environment.GUI;
import net.javacomm.client.resource.Resource;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.Multilingual;
import net.javacomm.multilingual.schema.COMMAND;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.multilingual.schema.LANGUAGE;
import net.javacomm.multilingual.schema.LANGUAGES;
import net.javacomm.multilingual.schema.VIRTUALKEY;
import net.javacomm.window.manager.Control;
import net.javacomm.window.manager.WMResource;



/**
 * Der ServiceButton ist ein konfigurierbares 2-Button-Panel.
 *
 *
 * @author llange
 *
 */
public class MultilingualServiceButton extends JPanel implements Babelfish {

  private static final long serialVersionUID = -1242255510416830099L;
  private GridLayout grid = new GridLayout(1, 2, 24, 0);
  @SuppressWarnings("serial")
  protected JButton button_1 = new JButton() {
    @Override
    public JToolTip createToolTip() {
      ToolTipManager.sharedInstance().setInitialDelay(200);
      ToolTipManager.sharedInstance().setDismissDelay(1500);
      ArrowTooltip arrow = new ArrowTooltip(Resource.JQUERY_BLAU);
      arrow.setComponent(button_1);
      arrow.setTextAttributes(
          GUI.getInstance().registerFonts(Resource.TRUETYPE_REGULAR).deriveFont(Font.PLAIN, 13),
          Resource.JQUERY_HELLBLAU, Resource.JQUERY_BLAU
      );
      arrow.setBorder(new JToolTipBorder(7, Resource.JQUERY_YELLOW, Resource.JQUERY_BLAU));
      return arrow;
    }

  };
  @SuppressWarnings("serial")
  protected JButton button_2 = new JButton() {
    @Override
    public JToolTip createToolTip() {
      ToolTipManager.sharedInstance().setInitialDelay(200);
      ToolTipManager.sharedInstance().setDismissDelay(1500);
      ArrowTooltip arrow = new ArrowTooltip(Resource.JQUERY_BLAU);
      arrow.setComponent(button_2);
      arrow.setTextAttributes(
          WMResource.registerFonts(Resource.TRUETYPE_REGULAR).deriveFont(Font.PLAIN, 13),
          Resource.JQUERY_HELLBLAU, Resource.JQUERY_BLAU
      );
      arrow.setBorder(new JToolTipBorder(7, Resource.JQUERY_YELLOW, Resource.JQUERY_BLAU));
      return arrow;
    }

  };
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private ActionListener button1Listener = new ActionListener() {
    @Override
    public void actionPerformed(ActionEvent event) {
      changes.firePropertyChange(MultilingualServiceButton.class.getName(), Control.NULL, Control.BUTTON_1);
    }

  };
  private ActionListener button2Listener = new ActionListener() {
    @Override
    public void actionPerformed(ActionEvent event) {
      changes.firePropertyChange(MultilingualServiceButton.class.getName(), Control.NULL, Control.BUTTON_2);
    }

  };
  private KEY keyButton1;
  private KEY keyButton2;

  protected MultilingualServiceButton() {
    super(true);
    setLayout(grid);
    add(button_1);
    add(button_2);
    button_2.addActionListener(button2Listener);
  }



  /**
   *
   * @param button1
   *                ein Anzeigetext für Knopf 1
   * @param button2
   *                ein Anzeigetext für Knopf 2
   */
  protected MultilingualServiceButton(String button1, String button2) {
    this();
    button_1.setText(button1);
    button_2.setText(button2);
  }



  /**
   * Der bevorzugte Konstruktur für mehrsprachige Knöpfe.
   *
   * @param key1
   *             Knopf_1
   * @param key2
   *             Knopf_2
   */
  public MultilingualServiceButton(KEY key1, KEY key2) {
    this();
    keyButton1 = key1;
    keyButton2 = key2;
  }



  public void addButtonListener(PropertyChangeListener listener) {
    changes.addPropertyChangeListener(listener);
  }



  public void removeButtonListener(PropertyChangeListener listener) {
    changes.removePropertyChangeListener(listener);
  }



  public void removeAllListener() {
    for (PropertyChangeListener tmp : changes.getPropertyChangeListeners()) {
      changes.removePropertyChangeListener(tmp);
    }
    removeButton1Listener();
    removeButton2Listener();
  }



  /**
   * Der Listener von Knopf 1 wird entfernt.
   *
   */
  public synchronized void removeButton1Listener() {
    button_1.removeActionListener(button1Listener);
  }



  /**
   * Der Listener von Knopf 2 wird entfernt.
   *
   */
  public synchronized void removeButton2Listener() {
    button_2.removeActionListener(button2Listener);
  }



  /**
   * Lege für beide Knöpfe einen Shortcut fest.
   *
   *
   * @param menemonicButton1
   * @param menemonicButton1
   */
  public void setMnemonic(int menemonicButton1, int menemonicButton2) {
    button_1.setMnemonic(menemonicButton1);
    button_2.setMnemonic(menemonicButton2);
  }



  /**
   * Der Knopf_1 wird als außer Betrieb eingefärbt.
   *
   */
  public void setDisabledButton1() {
    removeButton1Listener();
    button_1.setForeground(WMResource.DISABLED_BUTTON_FOREGROUND);
    button_1.setBackground(WMResource.DISABLED_BUTTON_BACKGROUND);
  }



  public void setDisabledButton2() {
    removeButton2Listener();
    button_2.setForeground(WMResource.DISABLED_BUTTON_FOREGROUND);
    button_2.setBackground(WMResource.DISABLED_BUTTON_BACKGROUND);
  }



  /**
   * Knopf_1 nimmt erst jetzt Ereignisse entgegen. Der Knopf_1 wird als aktiv
   * eingefärbt.
   *
   */
  public void setEnabledButton1() {
    removeButton1Listener();
    button_1.addActionListener(button1Listener);
    button_1.setForeground(WMResource.ENABLED_BUTTON_FOREGROUND);
    button_1.setBackground(WMResource.ENABLED_BUTTON_BACKGROUND);
  }



  /**
   * Der Knopf_2 wird als aktiv eingefärbt.
   *
   */
  public void setEnabledButton2() {
    removeButton2Listener();
    button_2.addActionListener(button2Listener);
    button_2.setForeground(WMResource.ENABLED_BUTTON_FOREGROUND);
    button_2.setBackground(WMResource.ENABLED_BUTTON_BACKGROUND);
  }



  /**
   * Der Abstand zwischen den zwei Knöpfen.
   *
   * @param gap
   *            Abstand in Pixel
   */
  public void setHorizontalGap(int gap) {
    grid.setHgap(gap);
  }



  public String getButton_1_Text() {
    return button_1.getText();
  }



  public String getButton_2_Text() {
    return button_2.getText();
  }



  public void setIconButton1(Icon icon) {
    setIconButton1(icon, 4);
  }



  public void setIconButton1(Icon icon, int horizontalGap) {
    button_1.setIcon(icon);
    button_1.setIconTextGap(horizontalGap);
  }



  public void setIconButton2(Icon icon) {
    setIconButton2(icon, 4);
  }



  public void setIconButton2(Icon icon, int horizontalGap) {
    button_2.setIcon(icon);
    button_2.setIconTextGap(horizontalGap);
  }



  public void setMnemonicButton1(int mnemonic) {
    button_1.setMnemonic(mnemonic);
  }



  public void setMnemonicButton2(int mnemonic) {
    button_2.setMnemonic(mnemonic);
  }



  /**
   * Der Infotext von Knopf 1. Der Infotext wird ausgeblendet, wenn {@code null}
   * überegeben wird.
   *
   * @param text
   *             ein Hilfstext für den Anwender
   */
  public void setTooltipTextButton1(String text) {
    button_1.setToolTipText(text);
  }



  /**
   * Der Infotext von Knopf 2. Der Infotext wird ausgeblendet, wenn {@code null}
   * überegeben wird.
   *
   * @param text
   *             ein Hilfstext für den Anwender
   */
  public void setTooltipTextButton2(String text) {
    button_2.setToolTipText(text);
  }



  public void pressButton_1() {
    button_1.doClick();
  }



  /**
   * Die Sprache wird gesetzt.
   *
   * @param language
   *                 die Sprache
   */
  @Override
  public void setLanguage(ISO639 language) {
    setLanguage(keyButton1, keyButton2, language);
  }



  /**
   * Die Sprache darf jederzeit geändert werden.
   *
   * @param key
   *                 der Schlüssel für das ausgewählte Kommando
   * @param language
   *                 der Text wird in dieser Sprache angezeigt
   */
  public void setLanguage(KEY key1, KEY key2, ISO639 language) {
    if (key1 == null) throw new IllegalArgumentException("key1 ist null");
    if (key2 == null) throw new IllegalArgumentException("key2 ist null");
    if (language == null) throw new IllegalArgumentException("language ist null");
    this.keyButton1 = key1;
    this.keyButton2 = key2;
    COMMAND command = Multilingual.getInstance().getCommand(keyButton1);
    LANGUAGES languages = command.getLANGUAGES();
    if (languages == null) return;
    for (LANGUAGE tmp : languages.getLANGUAGE()) {
      if (tmp.getISO639() != language) continue;
      if (tmp.getTEXT() != null) {
        button_1.setText(tmp.getTEXT());
      }
      VIRTUALKEY vk = tmp.getSHORTCUT();
      if (vk != null) {
        button_1.setMnemonic(vk.ordinal() + 65);
      }
      String imagepath = tmp.getPathToImage();
      if (imagepath != null) {
        button_1.setIcon(new ImageIcon(getClass().getClassLoader().getResource(imagepath)));
      }
      break;
    }

    COMMAND command2 = Multilingual.getInstance().getCommand(keyButton2);
    LANGUAGES languages2 = command2.getLANGUAGES();
    if (languages2 == null) return;
    for (LANGUAGE tmp : languages2.getLANGUAGE()) {
      if (tmp.getISO639() != language) continue;
      if (tmp.getTEXT() != null) {
        button_2.setText(tmp.getTEXT());
      }
      VIRTUALKEY vk = tmp.getSHORTCUT();
      if (vk != null) {
        button_2.setMnemonic(vk.ordinal() + 65);
      }
      String imagepath = tmp.getPathToImage();
      if (imagepath != null) {
        button_2.setIcon(new ImageIcon(getClass().getClassLoader().getResource(imagepath)));
      }
      break;
    }

  }

}
