/*
 * Copyright © 2022 Luis Andrés Lange <http://javacomm.net>
 *
 * The MIT Licence
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
package net.javacomm.multilingual;

import javax.swing.JCheckBox;
import net.javacomm.multilingual.schema.COMMAND;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.multilingual.schema.LANGUAGE;
import net.javacomm.multilingual.schema.LANGUAGES;
import net.javacomm.multilingual.schema.VIRTUALKEY;



public class MultilingualCheckBox extends JCheckBox implements Babelfish {

  private static final long serialVersionUID = 858853237162951442L;

  private ISO639 language;
  private KEY key;

  protected MultilingualCheckBox() {
    super();
  }



  public MultilingualCheckBox(String text) {
    this();
    setText(text);
  }



  /**
   * Ein Menupunkt wird in deutscher Sprache erzeugt.
   *
   * @param key
   *            der Zugriffsschlüssel auf ein Menu
   */
  public MultilingualCheckBox(KEY key) {
    this(key, false);
  }



  public MultilingualCheckBox(KEY key, boolean state) {
    this(key, ISO639.DE);
    setSelected(state);
  }



  /**
   * Ein Menupunkt wird in einer bestimmten Sprache erzeugt.
   *
   * @param key
   *                 der Zugriffsschlüssel auf ein Menu
   * @param language
   *                 die angezeigte Sprache
   */
  public MultilingualCheckBox(KEY key, ISO639 language) {
    this();
    setLanguage(key, language);
  }



  /**
   * Der Menuschlüssel für die Textanzeige wird gesetzt.
   *
   * @param key
   *            der Zugriffschlüssel für JMenu, JMenuItem und JCheckBoxMenuItem in
   *            der Oberfläche
   *
   */
  public void setKey(KEY key) {
    setLanguage(key, language);
  }



  /**
   * Die Sprache wird gesetzt.
   *
   * @param language
   *                 die Sprache
   */
  @Override
  public void setLanguage(ISO639 language) {
    setLanguage(this.key, language);
  }



  /**
   * Die Sprache darf jederzeit geändert werden.
   *
   * @param key
   *                 der Schlüssel für das ausgewählte Kommando
   * @param language
   *                 der Text wird in dieser Sprache angezeigt
   */
  public void setLanguage(KEY key, ISO639 language) {
    if (key == null) throw new IllegalArgumentException("key ist null");
    if (language == null) throw new IllegalArgumentException("language ist null");
    this.language = language;
    this.key = key;
    COMMAND command = Multilingual.getInstance().getCommand(this.key);
    LANGUAGES languages = command.getLANGUAGES();
    if (languages == null) return;
    for (LANGUAGE tmp : languages.getLANGUAGE()) {
      if (tmp.getISO639() != language) continue;
      if (tmp.getTEXT() != null) {
        setText(tmp.getTEXT());
      }
      VIRTUALKEY vk = tmp.getSHORTCUT();
      if (vk != null) {
        setMnemonic(vk.ordinal() + 65);
      }
      break;
    }
  }

}
